$(function() {
	var $parentRow = $('#use-gps').parentsUntil('form');

	// disable 'Use My Location' button if geolocation functions are absent
	if (!(navigator && navigator.geolocation
		&& navigator.geolocation.getCurrentPosition)) {
		console.log('geolocation not available; removing location fields');
		$parentRow.prev().remove();
		$parentRow.remove();
	}

	$('#edit-all').click(editStreetClicked);
	$('#top').click(function() { $(document).scrollTop(0); });
	$('#to-search').click(function() {
		$(document).scrollTop($('.search-results').offset().top);
	});
	$('#select-left,#select-right,#select-inverse').click(modifySelection);
	
	$('#use-gps').click(function() {
		navigator.geolocation.getCurrentPosition(function(location) {
			$.setGeoLocData({
				latitude: {
					field: $('#latitude'),
					value: location.coords.latitude
				},
				longitude: {
					field: $('#longitude'),
					value: location.coords.longitude
				}
			});
			formChanged(false);
		});
	});

	// get all light poles once, so we don't need to hit the server every time
	// we'll still hit the server for GPS searches, since SQL has built-in
	// distance functions
	$.getJSON('//iot.lithl.info:1880/search', {})
			.done(function(data) {
		$.each(data, function(index, result) {
			result.pos = new google.maps.LatLng(result.latitude, result.longitude);
			$.lightmap.markers[result.serial] = result;
		});

		$.lightmap.showMap($('#map'), {
			lat: 38.627,
			lng: -90.1954,
			zoom: 15,
			disableDoubleClickZoom: true,
			keyboardShortcuts: false,
			mapTypeControl: false,
			streetViewControl: false
		}, initializeSearch);
	});

	// Auto-search when form inputs change
	$('form').submit(function(event) { event.preventDefault(); });
	$('#street,#cross,#latitude,#longitude,#range').change(function() {
		switch ($(this).attr('id')) {
			case 'street':
			case 'cross':
				formChanged(true);
				break;
			case 'latitude':
			case 'longitude':
			case 'range':
				formChanged(false);
				break;
		}
	});
	$('[name=range_unit] + ul a').click(function(event) {
		event.preventDefault();
		$('[name=range_unit]').text($(this).text())
							  .val($(this).text())
							  .append($('<i>').addClass('caret'));
		formChanged(false);
	});
	
	// Size the map
	$(window).resize(function() {
		$('#map').width($('#map').parent().width())
		if ($(window).width() > 974) {
			$('#map').width(Math.min($('#map').width(), 423));
		}
		$('#map').height($('#map').width());
	}).resize();
});

$.extend({
	// search result coordinates, used for copys street results to map
	search: {
		coords: []
	}
});

function modifySelection(event) {
	var $self = $(this),
		btnId = $self.attr('id');
	
	event.preventDefault();
	$self.blur();
	$('#results-body tr').each(function(i, tr) {
		var $tr = $(tr);
		if (btnId === 'select-left') {
			if ($tr.data('pole') === 'left' && !$tr.hasClass('info')) {
				$tr.click();
			}
		} else if (btnId === 'select-right') {
			if ($tr.data('pole') === 'right' && !$tr.hasClass('info')) {
				$tr.click();
			}
		} else if (btnId === 'select-inverse') {
			$tr.click();
		}
	});
}

/**
 * Initialize the search results table. Mostly relevant if the user hits the
 * back button after editing a schedule.
 */
function initializeSearch() {
	var results = JSON.parse(sessionStorage.getItem('search-results'));
	
	$('#street,#cross,#latitude,#longitude,#range,[name=range_unit]').val(
		function() {
			var key;
			
			if ($(this).attr('id')) {
				key = $(this).attr('id');
			} else {
				key = 'range_unit';
			}
			
			return sessionStorage.getItem(key);
	});
	$('[name=range_unit]').text($('[name=range_unit]').val())
						  .append($('<i>').addClass('caret'));
	showSearchResults(results);
	showSearchResults(results, true);
}

/**
 * Store search form values in sessionStorage
 *
 * @param searchResults Array the actual results in the table
 */
function storeSearch(searchResults) {
	sessionStorage.setItem('search-results', JSON.stringify(searchResults));
	sessionStorage.setItem('street', $('#street').val());
	sessionStorage.setItem('cross', $('#cross').val());
	sessionStorage.setItem('latitude', $('#latitude').val());
	sessionStorage.setItem('longitude', $('#longitude').val());
	sessionStorage.setItem('range', $('#range').val());
	sessionStorage.setItem('range_unit', $('[name=range_unit]').val());
}

/**
 * Perform a search for light poles when form elements change.
 * This reduces user clicks by one, by not requiring them to
 * confirm the search query.
 *
 * @param byStreet boolean search by the #street and #cross select dropdowns or
 *                         by the #latitude, #longitude, and #range inputs
 */
function formChanged(byStreet) {
	var street_id = parseInt($('#street').val()),
		intersection_id = parseInt($('#cross').val()),
		latitude = parseFloat($('#latitude').val()),
		longitude = parseFloat($('#longitude').val()),
		range = parseInt($('#range').val()),
		range_unit = $('[name=range_unit]').val(),
		i, key, val, results = [];

	if (byStreet) {
		for (key in $.lightmap.markers) {
			val = $.lightmap.markers[key];
			if (intersection_id) {
				// street & cross
				if ((val.street_id === street_id
						&& val.intersection_id === intersection_id)
					|| (val.street_id === intersection_id
						&& val.intersection_id === street_id)) {
					results.push(key);
				}
			} else {
				// steet only
				if (val.street_id === street_id
					|| val.intersection_id === street_id) {
					results.push(key);
				}
			}
		}
		showSearchResults(results, true);
	} else {
		$.getJSON('//iot.lithl.info:1880/search', {
			longitude: longitude,
			latitude: latitude,
			range: range,
			range_unit: range_unit
		}).done(function(data) {
			for (i = 0; i < data.length; i++) {
				val = data[i];
				results.push(val.serial);
			}
			showSearchResults(results, true);
		});
	}
}

/**
 * Display the search results table, or hide it if there are no results.
 * This function will be called separately with `updateMap` as true and
 * false/undefined to avoid loops occuring with marker updates.
 *
 * @param results Array an array of the serial ids for the poles in the results
 * @param updateMap boolean whether to update the map's marker colors
 */
function showSearchResults(results, updateMap) {
	var $tbody = $('#results-body'),
		$tr,
		$td = $('<td>'),
		$tdSerialId, $tdStreetName, $tdCrossStreet, $tdLatitude,
		$tdLongitude, $tdActions, $actionButton,
		i, k, data, bounds,
		sw = { lat: Infinity, lng: Infinity },
		ne = { lat: -Infinity, lng: -Infinity };

	storeSearch(results);
	
	// Hide the results table and deselect all markers
	if (results.length === 0) {
		$('.search-results,#to-search').hide();
		if (updateMap) {
			for (k in $.lightmap.markers) {
				data = $.lightmap.markers[k];
				if (!data.marker) continue;
				data.marker.setIcon($.DESELECTED_ICON);
			}
		}
		return;
	}

	// Update the selection status of the map markers
	if (updateMap) {
		for (k in $.lightmap.markers) {
			data = $.lightmap.markers[k];
			if (!data.marker) continue;
			
			if (results.indexOf(k) >= 0) {
				data.marker.setIcon($.SELECTED_ICON);
				sw.lat = Math.min(sw.lat, data.latitude);
				sw.lng = Math.min(sw.lng, data.longitude);
				ne.lat = Math.max(ne.lat, data.latitude);
				ne.lng = Math.max(ne.lng, data.longitude);
			} else {
				data.marker.setIcon($.DESELECTED_ICON);
			}
		}
		$.lightmap.map.fitBounds(new google.maps.LatLngBounds(sw, ne));
		$.lightmap.map.panToBounds(new google.maps.LatLngBounds(sw, ne));
		return;
	}

	// Show & populate the results table
	$tbody.empty();
	results.sort();	

	for (i = 0; i < results.length; i++) {
		data = $.lightmap.markers[results[i]];
		$tr = $('<tr>');
		$tdSerialId = $td.clone();
		$tdStreetName = $td.clone();
		$tdCrossStreet = $td.clone();
		$tdLatitude = $td.clone();
		$tdLongitude = $td.clone();
		$tdActions = $td.clone();
		$actionButton = $('#action-template > button').clone();
	
		$tdSerialId.text(data.serial);
		$tdStreetName.text(data.street_name);
		if (data.cross_street == data.street_name) {
			$tdCrossStreet.text(data.cross_alternate);
		} else {
			$tdCrossStreet.text(data.cross_street);
		}
		$tdLatitude.text(data.latitude);
		$tdLongitude.text(data.longitude);
		$actionButton.attr({ 'data-id': data.id });
		
		$tdActions.append($actionButton);
		
		$actionButton.click(editStreetClicked);
		$tr.data('pole', data.side);
		$tr.click(toggleStreetSelected);
		
		$tr.append($tdActions, $tdSerialId, $tdStreetName, $tdCrossStreet,
					$tdLatitude, $tdLongitude);
		$tbody.append($tr);
	}
	$('.search-results,#to-search').show();
}

/**
 * Search by map selection. Leave a delay before actually updating the
 * search, because this function will be called multiple times for a
 * drag+select multiple.
 *
 * @param latLng google.maps.LatLng the map marker location to update with
 * @param include boolean whether the point is being added to the collection of
 *                        selected markers or not
 */
function updateSearchResultsWithLatLng(latLng, include) {
	var strcpy, i;
	
	for (i = 0; i < $.search.coords.length; i++) {
		if (Math.abs($.search.coords[i].lat - latLng.lat()) < 0.00001
			&& Math.abs($.search.coords[i].lng - latLng.lng()) < 0.00001) {
			$.search.coords.splice(i, 1);
		}
	}

	if (include) {
		$.search.coords.push({ lat: latLng.lat(), lng: latLng.lng() });
	}
	strcpy = JSON.stringify($.search.coords);
	
	setTimeout((function(cpy) {
		return function() {
			var strcrd = JSON.stringify($.search.coords),
				results = [],
				i, latLng, key, marker;
			if (cpy !== strcrd) return;

			for (i = 0; i < $.search.coords.length; i++) {
				latLng = $.search.coords[i];

				for (key in $.lightmap.markers) {
					marker = $.lightmap.markers[key];

					if (Math.abs(marker.latitude - latLng.lat) < 0.00001
						&& Math.abs(marker.longitude - latLng.lng) < 0.00001) {
						results.push(key);
					}
				}
			}

			showSearchResults(results);
		};
	})(strcpy), 50);
}

/**
 * Send the user to the edit schedule page with appropriate querystring
 * when any of the edit schedule buttons (including the edit selected button)
 * is clicked.
 */
function editStreetClicked() {
	var $lights, query = '?';

	if ($(this).is('#edit-all')) {
		$lights = $(this).parents('tfoot')  // tfoot
						 .prev()            // tbody
						 .children('.info') // tr
						 .find('button');   // button
	} else {
		$lights = $(this);
	}

	query += $lights.map(function(index, light) {
			return 'ids[]=' + $(light).data('id');
		}).toArray().join('&');
	location.href = location.protocol + '//' + location.host + '/edit' + query;
}

/**
 * Toggle highlight on rows in the search results the user clicks on
 * then, enable/disable the 'Edit All' button at the bottom of the table
 * if any rows are highlighted.
 */
function toggleStreetSelected() {
	var $row = $(this),
		$allRows = $row.parent().children(),
		$editAllButton = $('#edit-all'),
		anySelected = false;

	if ($(this)[0].nodeName.toLowerCase() === 'button') return;

	$row.toggleClass('info');
	$.each($allRows, function(index, row) {
		anySelected |= $(row).hasClass('info');
	});
	if (anySelected) {
		$editAllButton.removeAttr('disabled');
	} else {
		$editAllButton.attr('disabled', 'disabled');
	}
}
